# src/vol5_k2m_cc/epsilon_soften.py
# -*- coding: utf-8 -*-
"""
Safe parser for Poisson softening expressions.

Allowed forms for epsilon_soften:
  - a plain number (e.g., 0.75, "1", "1.0")
  - "sigma"
  - "c*sigma" where c is a real number (e.g., "1.0*sigma", "0.5 * sigma")

This avoids eval() and fixes crashes like: ValueError: could not convert string to float: '1.0*4.0'
"""

from __future__ import annotations
import re
from typing import Union

Number = Union[int, float]

# Matches either a plain number OR an optional coefficient times the literal 'sigma'
_SIGMA_PAT = re.compile(
    r"""
    ^\s*
    (?:
        (?P<num>[-+]?(?:\d+(?:\.\d*)?|\.\d+))     #  number like 1, 1.0, .5
      |
        (?:(?P<coef>[-+]?(?:\d+(?:\.\d*)?|\.\d+))\s*\*\s*)?
        (?P<sigma>sigma)                           #  the literal word 'sigma'
    )
    \s*$
    """,
    re.IGNORECASE | re.VERBOSE,
)

def parse_epsilon_soften(expr: Union[str, Number, None], sigma_max: float) -> float:
    """
    Parse epsilon_soften and return a float.

    Parameters
    ----------
    expr : str | float | int | None
        One of: number, "sigma", or "c*sigma".
    sigma_max : float
        The reference sigma value (usually max(sigma_list)).

    Returns
    -------
    float
        The softening epsilon.

    Raises
    ------
    ValueError
        If the expression is not in an allowed form.
    """
    if expr is None:
        return float(sigma_max)  # sensible default
    if isinstance(expr, (int, float)):
        return float(expr)

    s = str(expr).strip()
    m = _SIGMA_PAT.match(s)
    if not m:
        raise ValueError(
            f"epsilon_soften must be a number, 'sigma', or 'c*sigma'; got {expr!r}"
        )

    if m.group("sigma"):  # "sigma" or "c*sigma"
        coef = m.group("coef")
        c = float(coef) if coef is not None else 1.0
        return c * float(sigma_max)

    # plain number
    return float(m.group("num"))


__all__ = ["parse_epsilon_soften"]


if __name__ == "__main__":
    # quick self-test
    tests = [None, 0.5, "1", "1.0", "sigma", "1*sigma", "0.5*sigma", " 1.5 * sigma "]
    for t in tests:
        print(f"{t!r}  ->  {parse_epsilon_soften(t, sigma_max=4.0)}")
